/* properties.c - properties box for xpenguins_applet
 * Copyright (C) 1999-2001  Robin Hogan
 * Copyright (C) 2002       Sebastian Henschel
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************************************************
 *  preprocessor
 ****************************************************************************/

#include <config.h>
#include <stdio.h>
#include <gnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>

#include "xpenguins.h"
#include "applet.h"



/*****************************************************************************
 *  declarations
 ****************************************************************************/

static GtkWidget * make_page_general (XPenguinsApplet *);
static GtkWidget * make_page_themes (XPenguinsApplet *);
static void set_atk_name_desc (GtkWidget *, const gchar *, const gchar *);
static void set_relation (GtkWidget *, GtkLabel *);
static void destroy (GtkWidget *, gpointer);
static void response (GtkDialog *, gint, gpointer);
static void help ();
static void restart_timer (XPenguinsApplet *);
static void set_theme_error_label (XPenguinsApplet *, const gchar *,
                                   const gchar *);
static void set_theme_number_speed (XPenguinsApplet *);
static void update_overridetheme (XPenguinsApplet *, gboolean);
static void update_number (XPenguinsApplet *, gint);
static void update_speed (XPenguinsApplet *, gint);
static void update_ignorepopups (XPenguinsApplet *, gboolean);
static void update_noblood (XPenguinsApplet *, gboolean);
static void update_noangels (XPenguinsApplet *, gboolean);
static void update_theme_widgets (XPenguinsApplet *);
static gboolean update_theme (XPenguinsApplet *, const gchar *);
static void cb_update (GtkWidget *, gpointer);
static void cb_update_theme (GtkTreeSelection *, gpointer);

static const gchar *key_themename = "theme_name";
static const gchar *key_overridetheme = "override_theme";
static const gchar *key_number = "number";
static const gchar *key_speed = "speed";
static const gchar *key_noblood = "no_blood";
static const gchar *key_noangels = "no_angels";
static const gchar *key_ignorepopups = "ignore_popups";

static const gint default_speed = 20;
static const gint default_divident = 1000;


/*****************************************************************************
 *  extern functions
 ****************************************************************************/

/* Load the current settings from gconf */
extern void
xpa_property_load (XPenguinsApplet *xpa)
{
    gchar *theme_name;

    theme_name = panel_applet_gconf_get_string (xpa->applet, key_themename,
                                          NULL);
    if (theme_name == NULL) theme_name = g_strdup ("Penguins");

    update_theme (xpa, theme_name);

    update_overridetheme (xpa, panel_applet_gconf_get_bool (
                          xpa->applet, key_overridetheme, NULL));
    update_number (xpa, panel_applet_gconf_get_int (
                   xpa->applet, key_number, NULL));
    update_speed (xpa, panel_applet_gconf_get_int (
                  xpa->applet, key_speed, NULL));
    update_ignorepopups (xpa, panel_applet_gconf_get_bool (
                         xpa->applet, key_ignorepopups, NULL));
    update_noblood (xpa, panel_applet_gconf_get_bool (
                    xpa->applet, key_noblood, NULL));
    update_noangels (xpa, panel_applet_gconf_get_bool (
                     xpa->applet, key_noangels, NULL));

    g_free (theme_name);
}



/* Create the property box, from which the properties of xpenguins can
 * be changed */
extern void
xpa_property_show (BonoboUIComponent *uic, XPenguinsApplet *xpa,
                   const gchar *verbname)
{
    GtkWidget *notebook;
    GtkWidget *page;
    GtkWidget *label;

    /* If property box already exists, bring it to the top and return */
    if (xpa->prop_window)
    {
        gtk_window_present (GTK_WINDOW (xpa->prop_window));
        return;
    }

    /* Make a new property window */
    xpa->prop_window = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW (xpa->prop_window),
                          _("XPenguins Applet properties"));

    /* Add the buttons */
    gtk_dialog_add_button (GTK_DIALOG (xpa->prop_window),
                           GTK_STOCK_HELP, GTK_RESPONSE_HELP);
    gtk_dialog_add_button (GTK_DIALOG (xpa->prop_window),
                            GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
    gtk_dialog_set_default_response (GTK_DIALOG (xpa->prop_window),
                                     GTK_RESPONSE_CLOSE);

    /* Property box contains two pages: "General" and "Themes" */
    notebook = gtk_notebook_new ();

    /* First the "General" page, which is a vertical container */
    page = make_page_general (xpa);
    /* Post the "General" page in the property box */
    label = gtk_label_new_with_mnemonic (_("_General"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);
    gtk_widget_show (page);

    /* Now the "Themes" page, again a vertical container */
    page = make_page_themes (xpa);
    /* Post the "Themes" page in the property box */
    label = gtk_label_new_with_mnemonic (_("_Themes"));
    gtk_notebook_append_page (GTK_NOTEBOOK(notebook), page, label);
    gtk_widget_show (page);

    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (xpa->prop_window)->vbox),
                        notebook, TRUE, TRUE, 0);
    gtk_widget_show (notebook);

    /* Callbacks for the various property-box buttons */
    g_signal_connect (G_OBJECT (xpa->prop_window), "response",
                      G_CALLBACK (response), xpa);
    g_signal_connect (G_OBJECT(xpa->prop_window), "destroy",
                      G_CALLBACK (destroy), xpa);

    gtk_widget_grab_focus (xpa->overridetheme_check);
    gtk_widget_show (xpa->prop_window);
}



/*****************************************************************************
 *  static functions
 ****************************************************************************/

/* Make the general page of the notebook */
static GtkWidget *
make_page_general (XPenguinsApplet *xpa)
{
    GtkWidget *page;
    GtkWidget *range;
    GtkWidget *label;
    GtkWidget *evtbox;
    GtkWidget *table;
    GtkTooltips *tooltip = NULL;
    GError *error;

    tooltip = gtk_tooltips_new ();
    page = gtk_vbox_new (FALSE, GNOME_PAD);
    gtk_container_set_border_width (GTK_CONTAINER (page), GNOME_PAD);


    /* Override the number and speed of toons in the current theme */
    xpa->overridetheme_check = gtk_check_button_new_with_mnemonic (
        _("_Override number and speed of toons in theme"));
    gtk_tooltips_set_tip (tooltip, GTK_WIDGET (xpa->overridetheme_check),
                          _("Shall the values for the number of toons and the "
                            "frame rate provided by the theme be overriden by "
                            "the user?"), NULL);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (xpa->overridetheme_check),
                                  xpa->overridetheme);
    gtk_box_pack_start_defaults (GTK_BOX (page), xpa->overridetheme_check);
    g_signal_connect (G_OBJECT (xpa->overridetheme_check), "clicked",
                      G_CALLBACK (cb_update), xpa);
    gtk_widget_show (xpa->overridetheme_check);

    /* The override button controls this table, which contains the
     * number and speed override controls */
    xpa->table = table = gtk_table_new (2, 2, FALSE);
    gtk_box_pack_start_defaults (GTK_BOX (page), table);
    gtk_widget_show (table);
    gtk_widget_set_sensitive (GTK_WIDGET (xpa->table), xpa->overridetheme);

    /* Override the number of penguins */
    /* First the label */
    evtbox = gtk_event_box_new ();
    gtk_tooltips_set_tip (tooltip, evtbox,
                          _("How many toons shall be drawn?"), NULL);
    label = gtk_label_new_with_mnemonic (_("Number of toons:"));
    gtk_container_add (GTK_CONTAINER (evtbox), label);
    gtk_widget_show (label);
    gtk_table_attach (GTK_TABLE (table), evtbox, 0, 1, 0, 1,
                      GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 5, 5);
    gtk_widget_show (evtbox);
    /* Then the slider */
    xpa->number_adj = (GtkAdjustment *) gtk_adjustment_new (xpa->number,
                                                            1, 110, 1, 1, 10);
    range = gtk_hscale_new (GTK_ADJUSTMENT (xpa->number_adj));
    gtk_scale_set_digits (GTK_SCALE (range), 0);
    gtk_table_attach (GTK_TABLE (table), range, 1, 2, 0, 1,
                      GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_SHRINK, 5, 5);
    g_signal_connect (G_OBJECT (xpa->number_adj), "value_changed",
                      G_CALLBACK (cb_update), xpa);
    gtk_range_set_update_policy (GTK_RANGE (range), GTK_UPDATE_DISCONTINUOUS);
    set_relation (range, GTK_LABEL (label));
    gtk_widget_show (range);

    /* Override the frame rate */
    /* First the label */
    evtbox = gtk_event_box_new ();
    gtk_tooltips_set_tip (tooltip, evtbox,
                          _("How fast shall the toons move?"),
                          NULL);
    label = gtk_label_new_with_mnemonic (_("Frame rate (Hz):"));
    gtk_container_add (GTK_CONTAINER (evtbox), label);
    gtk_widget_show (label);
    gtk_table_attach (GTK_TABLE (table), evtbox, 0, 1, 1, 2,
                      GTK_FILL | GTK_SHRINK , GTK_FILL | GTK_SHRINK, 5, 5);
    gtk_widget_show (evtbox);
    /* Then the slider */
    xpa->speed_adj = (GtkAdjustment *) gtk_adjustment_new (xpa->speed,
                                                           1, 110, 1, 1, 10);
    range = gtk_hscale_new (GTK_ADJUSTMENT (xpa->speed_adj));
    gtk_scale_set_digits (GTK_SCALE (range), 0);
    gtk_table_attach (GTK_TABLE (table), range, 1, 2, 1, 2,
                      GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_SHRINK, 5, 5);
    g_signal_connect (G_OBJECT (xpa->speed_adj), "value_changed",
                      G_CALLBACK (cb_update), xpa);
    gtk_range_set_update_policy (GTK_RANGE (range), GTK_UPDATE_DISCONTINUOUS);
    set_relation (range, GTK_LABEL (label));
    gtk_widget_show (range);

    /* Some check buttons: first one for ignoring popup windows */
    xpa->ignorepopups_check = gtk_check_button_new_with_mnemonic (
                              _("_Ignore popup windows"));
    gtk_tooltips_set_tip (tooltip, GTK_WIDGET (xpa->ignorepopups_check),
                          _("Shall popup-windows be ignored by the toons?"),
                          NULL);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (xpa->ignorepopups_check),
                                  xpa->ignorepopups);
    gtk_box_pack_start_defaults (GTK_BOX (page), xpa->ignorepopups_check);
    g_signal_connect (G_OBJECT (xpa->ignorepopups_check), "clicked",
                      G_CALLBACK (cb_update), xpa);
    gtk_widget_show (xpa->ignorepopups_check);

    /* Then one for omitting all the gory images! */
    xpa->noblood_check = gtk_check_button_new_with_mnemonic (
                         _("Do not show gory _death sequences"));
    gtk_tooltips_set_tip (tooltip, GTK_WIDGET (xpa->noblood_check),
                          _("Do not show gory death sequences when toons "
                            "are squished?"), NULL);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (xpa->noblood_check),
                                  xpa->noblood);
    gtk_box_pack_start_defaults (GTK_BOX (page), xpa->noblood_check);
    g_signal_connect (G_OBJECT (xpa->noblood_check), "clicked",
                      G_CALLBACK (cb_update), xpa);
    gtk_widget_show (xpa->noblood_check);

    /* Then one for omitting all the angels! */
    xpa->noangels_check = gtk_check_button_new_with_mnemonic (
                         _("Do not show _angels"));
    gtk_tooltips_set_tip (tooltip, GTK_WIDGET (xpa->noangels_check),
                          _("Do not show angels after dying?"), NULL);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (xpa->noangels_check),
                                  xpa->noangels);
    gtk_box_pack_start_defaults (GTK_BOX (page), xpa->noangels_check);
    g_signal_connect (G_OBJECT (xpa->noangels_check), "clicked",
                      G_CALLBACK (cb_update), xpa);
    gtk_widget_show (xpa->noangels_check);

    return page;
}



/* Make the themes page of the notebook */
static GtkWidget *
make_page_themes (XPenguinsApplet *xpa)
{
    GtkWidget *page;
    GtkWidget *scroll;
    GtkListStore *storage;
    GtkCellRenderer *cell;
    GtkTreeViewColumn *column;

    page = gtk_vbox_new (FALSE, GNOME_PAD);
    gtk_container_set_border_width (GTK_CONTAINER (page), GNOME_PAD);

    /* Preview themes in a horizontal container in which we put the
     * icon and the text information
     */
    xpa->preview = gtk_hbox_new (FALSE, GNOME_PAD);
    xpa->info = gtk_label_new (NULL);
    xpa->preview_pixmap = NULL;
    set_atk_name_desc (xpa->preview_pixmap, _("Preview Pixmap"),
                       _("A small pixmap to preview the look of the theme"));
    gtk_label_set_justify (GTK_LABEL (xpa->info), GTK_JUSTIFY_LEFT);
    gtk_box_pack_end_defaults (GTK_BOX (xpa->preview), xpa->info);
    gtk_widget_show (xpa->info);

    /* Prevent hopping of tree view when preview image has different heights */
    gtk_widget_set_size_request (xpa->preview, -1, 100);
    gtk_box_pack_start (GTK_BOX (page), xpa->preview, FALSE, FALSE, 0);
    gtk_widget_show (xpa->preview);

    /* Update widgets to show current theme */
    update_theme_widgets (xpa);

    /* The last widget is a scrolled window containing a list of
     * the available themes
     */
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
    xpa->treeview = (GtkTreeView *) gtk_tree_view_new ();
    gtk_widget_set_size_request (GTK_WIDGET (xpa->treeview), -1, 100);
    storage = gtk_list_store_new (1, G_TYPE_STRING);
    gtk_tree_view_set_model (xpa->treeview, GTK_TREE_MODEL (storage));

    /* Clear the theme list from a previous call */
    if (xpa->theme_list)
    {
        xpenguins_free_list (xpa->theme_list);
        xpa->theme_list = NULL;
    }

    /* Call the xpenguins library to find the available themes */
    xpa->theme_list = xpenguins_list_themes (&(xpa->nthemes));

    /* Append each theme to the list */
    if (xpa->theme_list)
    {
        GtkTreeIter iter, cur_iter;
        gboolean cur_found = FALSE;
        gint i;

        for (i = 0; i < xpa->nthemes; ++i)
        {
            gtk_list_store_append (storage, &iter);
            gtk_list_store_set (storage, &iter, 0, xpa->theme_list[i], -1);

            /* See if the current theme is in the list */
            if (strcmp (xpa->theme_list[i], xpa->theme_name) == 0)
            {
                cur_iter = iter;
                cur_found = TRUE;
            }
        }

        cell = gtk_cell_renderer_text_new ();
        column = gtk_tree_view_column_new_with_attributes (_("_Select Theme"),
                                                           cell, "text", 0,
                                                           NULL);
        gtk_tree_view_column_set_alignment (column, 0.5);
        gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (storage), 0,
                                              GTK_SORT_ASCENDING);
        gtk_tree_view_column_set_sort_column_id (column, 0);
        gtk_tree_view_column_set_sort_indicator (column, TRUE);
        gtk_tree_view_append_column (xpa->treeview, column);

        /* Activate entry corresponding to current theme */
        if (cur_found)
        {
            GtkTreePath *path;

            path = gtk_tree_model_get_path (
                       gtk_tree_view_get_model (xpa->treeview),
                       &cur_iter);
            gtk_tree_view_set_cursor (xpa->treeview, path, column, FALSE);
            gtk_tree_view_scroll_to_cell (xpa->treeview, path, NULL,
                                          TRUE, 0.5, 0.5);
            g_free (path);
        }

        xpa->treeview_signal_id = g_signal_connect (
            G_OBJECT (gtk_tree_view_get_selection (xpa->treeview)),
            "changed", G_CALLBACK (cb_update_theme), xpa);
  }
    else /* No themes found! */
    {
        set_theme_error_label (xpa, XPENGUINS_SYSTEM_DIRECTORY,
                               _("no themes found\nMake sure you have "
                                 "properly installed the 'xpenguins' "
                                 "package"));
    }

    /* Set list properties and pack */
    gtk_container_add (GTK_CONTAINER (scroll), GTK_WIDGET (xpa->treeview));
    gtk_widget_show (GTK_WIDGET (xpa->treeview));
    gtk_box_pack_start_defaults (GTK_BOX (page), scroll);
    gtk_widget_show (scroll);

    return page;
}



/* Sets the name and description of a widget for ATK */
static void
set_atk_name_desc (GtkWidget *widget, const gchar* name, const gchar *desc)
{
    AtkObject *obj;

    g_return_if_fail (widget != NULL);
    g_return_if_fail (name != NULL);
    g_return_if_fail (desc != NULL);

    obj = gtk_widget_get_accessible (widget);
    if (!GTK_IS_ACCESSIBLE (obj)) return;

    atk_object_set_name (obj, name);
    atk_object_set_description (obj, desc);
}



/* Set the relation between a widget and a label for ATK */
static void
set_relation (GtkWidget *widget, GtkLabel *label)
{
    AtkObject *atk_widget, *atk_label;
    AtkRelationSet *set;
    AtkRelation *relation;

    g_return_if_fail (widget != NULL);
    g_return_if_fail (label != NULL);

    /* Set the ATK_RELATION_LABEL_FOR relation */
    gtk_label_set_mnemonic_widget (label, widget);

    atk_widget = gtk_widget_get_accessible (widget);
    if (!GTK_IS_ACCESSIBLE (atk_widget)) return;

    atk_label = gtk_widget_get_accessible (GTK_WIDGET (label));
    relation = atk_relation_new (&atk_label, 1, ATK_RELATION_LABELLED_BY);
    set = atk_object_ref_relation_set (atk_widget);
    atk_relation_set_add (set, relation);

    g_object_unref (relation);
}



/* Callback to destroy properties */
static void
destroy (GtkWidget *widget, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    gtk_widget_destroy (xpa->prop_window);
    xpa->prop_window = NULL;
}



/* Callback to respond to click on help/apply/ok/close */
static void
response (GtkDialog *dialog, gint id, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    switch (id)
    {
        case GTK_RESPONSE_HELP:
            help ();
            break;
        case GTK_RESPONSE_CLOSE:
        default:
            destroy (GTK_WIDGET (dialog), xpa);
            break;
    }
}



/* Display help dialog */
static void
help ()
{
    GError *error = NULL;

    gnome_help_display_desktop (NULL, PACKAGE, PACKAGE,
                                "props", &error);
    if (error)
    {
        g_warning ("help error: %s\n", error->message);
        g_error_free (error);
    }
}



static void
restart_timer (XPenguinsApplet *xpa)
{
     gtk_timeout_remove (xpa->timeout);
     xpa->timeout = gtk_timeout_add (default_divident / xpa->speed,
                                        xpa_service_xpenguins, xpa);
}



/* Convenience function for setting the info label on the themes page in case
 * of error
 */
static void
set_theme_error_label (XPenguinsApplet *xpa, const gchar* theme_name,
                       const gchar *msg)
{
    gchar *str = g_strdup_printf (_("theme error: %s (%s)"), theme_name, msg);

    gtk_label_set_text (GTK_LABEL (xpa->info), str);
    g_free (str);
    if (xpa->preview_pixmap) gtk_widget_destroy (xpa->preview_pixmap);
}



/* Sets the number and speed widgets/variables, etc. enforced by the theme */
static void
set_theme_number_speed (XPenguinsApplet *xpa)
{
    guint delay = (xpa->theme).delay;

    if (delay != 0) xpa->speed = default_divident / delay;
    else xpa->speed = default_speed;
    xpa->number = (xpa->theme).total;

    if (xpa->prop_window)
    {
        gtk_adjustment_set_value (GTK_ADJUSTMENT (xpa->number_adj),
                                  xpa->number);
        gtk_adjustment_set_value (GTK_ADJUSTMENT (xpa->speed_adj),
                                  xpa->speed);
    }
}



/* Updates the override theme settings */
static void
update_overridetheme (XPenguinsApplet *xpa, gboolean state)
{
    xpa->overridetheme = state;
    if (xpa->loaded)
    {
        if (xpa->overridetheme)
        {
            xpa->number = panel_applet_gconf_get_int (xpa->applet,
                                                key_number, NULL);
            xpa->speed = panel_applet_gconf_get_int (xpa->applet, key_speed,
                                               NULL);
        }
        else
            set_theme_number_speed (xpa);
        xpenguins_set_number (xpa->number);
    }
}


/* Updates the number of toons */
static void
update_number (XPenguinsApplet *xpa, gint number)
{
    xpa->number = number;
    xpenguins_set_number (number);
}


/* Updates the speed of the toons */
static void
update_speed (XPenguinsApplet *xpa, gint speed)
{
    xpa->speed = speed;
}



/* Updates the property ignorepopups */
static void
update_ignorepopups (XPenguinsApplet *xpa, gboolean state)
{
    xpa->ignorepopups = state;
    xpenguins_ignorepopups (state);
}



/* Updates the property noblood */
static void
update_noblood (XPenguinsApplet *xpa, gboolean state)
{
    xpa->noblood = state;
    xpenguins_set_blood (!state);
}



/* Updates the property noangels */
static void
update_noangels (XPenguinsApplet *xpa, gboolean state)
{
    xpa->noangels = state;
    xpenguins_set_angels (!state);
}



/* Updates the widgets of the theme */
static void
update_theme_widgets (XPenguinsApplet *xpa)
{
    if (xpa->theme_info) /* Clear old theme information */
    {
        xpenguins_free_list (xpa->theme_info);
        xpa->theme_info = NULL;
    }

    /* Get information about the new theme */
    if ((xpa->theme_info = xpenguins_theme_info (xpa->theme_name)))
    {
        /* Create the text information string - first add the theme name */
        GString *string = g_string_new (xpa->theme_name);
        if (xpa->preview_pixmap) gtk_widget_destroy (xpa->preview_pixmap);
        /* Display the icon if it exists */
        if ((xpa->preview_pixmap = gtk_image_new_from_file (
                                   xpa->theme_info[PENGUIN_ICON])))
        {
            gtk_box_pack_end (GTK_BOX (xpa->preview), xpa->preview_pixmap,
                              FALSE, FALSE, 5);
            gtk_widget_show (xpa->preview_pixmap);
        }
        if (*(xpa->theme_info[PENGUIN_DATE])) /* Add the date */
        {
            g_string_append (string, _(" (Last updated "));
            g_string_append (string, xpa->theme_info[PENGUIN_DATE]);
            g_string_append (string, ")");
        }
        if (*(xpa->theme_info[PENGUIN_ARTIST])) /* Add the artist(s) */
        {
            g_string_append (string, _("\nArtists: "));
            g_string_append (string, xpa->theme_info[PENGUIN_ARTIST]);
        }
        if (*(xpa->theme_info[PENGUIN_MAINTAINER])) /* Add the maintainer */
        {
            g_string_append (string, _("\nMaintainer: "));
            g_string_append (string, xpa->theme_info[PENGUIN_MAINTAINER]);
        }
        if (*(xpa->theme_info[PENGUIN_COPYRIGHT])) /* Add the copyright */
        {
            g_string_append (string, _("\nCopyright (C) "));
            g_string_append (string, xpa->theme_info[PENGUIN_COPYRIGHT]);
        }
        if (*(xpa->theme_info[PENGUIN_COMMENT])) /* Add the comment field */
        {
            g_string_append (string, "\n");
            g_string_append (string, xpa->theme_info[PENGUIN_COMMENT]);
        }
        /* Display the information */
        gtk_label_set_text (GTK_LABEL (xpa->info), string->str);
        g_string_free (string, FALSE);
    }
    else /* No "about" file found for theme */
    {
        set_theme_error_label (xpa, xpa->theme_name,
                               _("no information available"));
    }
}



/* Updates the theme */
static gboolean
update_theme (XPenguinsApplet *xpa, const gchar *next_theme_name)
{
    gchar *prev_theme_name;

    g_return_val_if_fail (next_theme_name != NULL, FALSE);
    if (xpa->theme_name
        && g_ascii_strncasecmp (xpa->theme_name, next_theme_name,
                                strlen (xpa->theme_name)) == 0)
        return FALSE;

    /* Stop xpenguins and free the old theme */
    if (xpa->active)
    {
        gtk_timeout_remove (xpa->timeout);
        xpenguins_exit ();
    }
    if (xpa->loaded)
    {
        xpenguins_free_theme (&(xpa->theme));
        xpa->loaded = FALSE;
    }

    /* Load new theme or re-load old theme */
    prev_theme_name = xpa->theme_name;
    xpa->theme_name = g_strdup (next_theme_name);
    if (!xpa_load_theme (xpa)) /* Reload old theme */
    {
        xpa->active = FALSE;
        g_free (xpa->theme_name);
        xpa->theme_name = prev_theme_name;
        xpa_load_theme (xpa);
        return FALSE;
    }
    else
        g_free (prev_theme_name);
    xpa->loaded = TRUE;

    /* Set number and speed if appropriate */
    if (!xpa->overridetheme) set_theme_number_speed (xpa);
    xpenguins_set_number (xpa->number);

    /* Update widgets */
    if (xpa->prop_window) update_theme_widgets (xpa);

    /* Start xpenguins and the service-timer */
    if (xpa->active)
    {
        gchar *error = xpenguins_start (NULL);
        if (error != NULL)
        {
            set_theme_error_label (xpa, xpa->theme_name, error);
            return FALSE;
        }
        restart_timer (xpa);
    }

    return TRUE;
}



/* Apply all the settings in the property box */
static void
cb_update (GtkWidget *widget, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    if (widget == xpa->overridetheme_check)
    {
        gboolean state;

        state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
                                                    xpa->overridetheme_check));
        panel_applet_gconf_set_bool (xpa->applet, key_overridetheme,
                               state, NULL);
        gtk_widget_set_sensitive (GTK_WIDGET (xpa->table), state);
        update_overridetheme (xpa, state);
    }

    else if (widget == GTK_WIDGET (xpa->number_adj))
    {
        gint val = gtk_adjustment_get_value (GTK_ADJUSTMENT (xpa->number_adj));
        panel_applet_gconf_set_int (xpa->applet, key_number, val, NULL);
        update_number (xpa, val);
    }

    else if (widget == GTK_WIDGET (xpa->speed_adj))
    {
        gint val = gtk_adjustment_get_value (GTK_ADJUSTMENT (xpa->speed_adj));
        panel_applet_gconf_set_int (xpa->applet, key_speed, val, NULL);
        update_speed (xpa, val);
    }

    else if (widget == xpa->ignorepopups_check)
    {
        gboolean state = gtk_toggle_button_get_active (
                    GTK_TOGGLE_BUTTON (xpa->ignorepopups_check));
        panel_applet_gconf_set_bool (
                xpa->applet, key_ignorepopups, state, NULL);
        update_ignorepopups (xpa, state);
    }

    else if (widget == xpa->noblood_check)
    {

       gboolean state = gtk_toggle_button_get_active (
                           GTK_TOGGLE_BUTTON (xpa->noblood_check));
        panel_applet_gconf_set_bool (xpa->applet, key_noblood, state, NULL);
       update_noblood (xpa, state);
    }

    else if (widget == xpa->noangels_check)
    {
        gboolean state = gtk_toggle_button_get_active (
                      GTK_TOGGLE_BUTTON (xpa->noangels_check));
        panel_applet_gconf_set_bool (xpa->applet, key_noangels, state, NULL);
        update_noangels (xpa, state);
    }

    if (xpa->active) restart_timer (xpa);
}



/* A theme has been selected in the theme list - display the preview
 * icon and text for it */
static void
cb_update_theme (GtkTreeSelection *selection, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;
    gchar *theme_name;
    GtkTreeIter iter;
    GtkTreeModel *model;

    gtk_tree_selection_get_selected (selection, &model, &iter);
    gtk_tree_model_get (model, &iter, 0, &theme_name, -1);

    if (update_theme (xpa, theme_name))
    {
        panel_applet_gconf_set_string (xpa->applet, key_themename,
                                 xpa->theme_name, NULL);
    }
    g_free (theme_name);
}

