/* XPENGUINS_APPLET - gnome2 panel implementation of xpenguins
 * Copyright (C) 1999-2001  Robin Hogan
 * Copyright (C) 2002       Sebastian Henschel
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Robin Hogan <R.J.Hogan@reading.ac.uk>
 * Sebastian Henschel <shensche@kodeaffe.de>
 * Project started 22 November 1999
 * This file last updated 5 November 2002
 */

/*****************************************************************************
 * preprocessor
 ****************************************************************************/

#include <config.h>
#include <gnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>

#include "sitter_button.xpm"

/* Button pixmaps look much nicer at small sizes if custom pixmaps are
 * used rather than just scaling down */
#include "play-pause-button.xpm"
#include "play-pause-button-small.xpm"
#include "play-pause-button-tiny.xpm"
#include "stop-button.xpm"
#include "stop-button-small.xpm"
#include "stop-button-tiny.xpm"

#include "xpenguins.h"
#include "applet.h"



/*****************************************************************************
 * declarations
 ****************************************************************************/

static void playpause (GtkWidget *, gpointer);
static void stop (GtkWidget *, gpointer);
static void destroy (GtkWidget *, gpointer);
static void help (BonoboUIComponent *, XPenguinsApplet *, const gchar *);
static void about (BonoboUIComponent *, XPenguinsApplet *, const gchar *);
static void make_pixmap_button (GtkWidget **, GtkWidget **, gchar **, 
                                gint, gint, guint);
static void change_size_cb (GtkWidget *, gint, gpointer);
static XPenguinsApplet *create_xpa (PanelApplet *);
static gboolean applet_fill (PanelApplet *);
static gboolean applet_factory(PanelApplet *, const gchar *,
                               gpointer);

/*****************************************************************************
 * static variables
 ****************************************************************************/

static const BonoboUIVerb menu_verbs[] = {
    BONOBO_UI_UNSAFE_VERB ("Properties", xpa_property_show),
    BONOBO_UI_UNSAFE_VERB ("Help",       help),
    BONOBO_UI_UNSAFE_VERB ("About",      about),
    BONOBO_UI_VERB_END
};

/* how to tell if the applet is already running?
 * xpenguins is not designed to run in multiple instances (see toon.h)
 * user is still allowed to start multiple instances but is warned about
 * the consequences.
 * any suggestions to do it better with gnome2-panel/bonobo are welcome
 */
static gint num_instances = 0;

/*****************************************************************************
 * extern functions
 ****************************************************************************/

/* Load theme and handle errors */
extern gboolean
xpa_load_theme (XPenguinsApplet *xpa)
{
    gchar *error;

    g_return_val_if_fail (xpa->theme_name != NULL, FALSE);
    g_return_val_if_fail (&(xpa->theme) != NULL, FALSE);

    error = xpenguins_load_theme (xpa->theme_name, &(xpa->theme));
    if (error)
    {
        GtkWidget *dialog;

        dialog = gtk_message_dialog_new (GTK_WINDOW (xpa->prop_window),
                                         GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_MESSAGE_ERROR,
                                         GTK_BUTTONS_CLOSE,
                                         _("Error loading theme '%s':\n %s"),
                                         xpa->theme_name, error);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);

        return (xpa->loaded = FALSE);
    }

    return (xpa->loaded = TRUE);
}


/* Timeout callback to update penguins */
extern gboolean
xpa_service_xpenguins (gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    if (xpa->paused)
        xpenguins_pause_frame ();
    else
    {
        if (!xpenguins_frame ())
        {
            /* No toons drawn! Must be end of exit sequence */
            gtk_timeout_remove (xpa->timeout);
            xpenguins_exit ();
            xpa->active = FALSE;
        }
    }

    return TRUE;
}



/*****************************************************************************
 * static functions
 ****************************************************************************/

/* Callback for play-pause button on the applet */
static void
playpause (GtkWidget *widget, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    if (!xpa->active)
    {
        xpa->paused = FALSE;
        if (!xpa->loaded)
        {
            /* Failed to load theme! */
            if (!xpa_load_theme (xpa)) return;
            xpa->loaded = TRUE;
            if (!xpa->overridetheme)
            {
                guint delay = (xpa->theme).delay;

                if (delay != 0) xpa->speed = 1000 / delay;
                else xpa->speed = 1;
                xpa->number = (xpa->theme).total;
            }
        }
        xpenguins_set_number (xpa->number);
        if (xpenguins_start (gdk_get_display ())) return;
        xpa->timeout =
            gtk_timeout_add (1000 / xpa->speed, xpa_service_xpenguins, xpa);
        xpa->active = TRUE;
    }
    else if (xpa->paused)
        xpa->paused = FALSE;
    else
        xpa->paused = TRUE;
}



/* Callback for the stop button on the applet */
static void
stop (GtkWidget *widget, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    if (xpa->active)
    {
        xpa->paused = FALSE;
        xpenguins_set_number (0);
    }
}



/* Exit cleanly and delete data */
static void
destroy (GtkWidget *widget, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    /* xpenguins */
    if (xpa->active) xpenguins_exit ();
    /* timeout */
    if (xpa->timeout != 0) gtk_timeout_remove (xpa->timeout);
    /* applet */
    g_free(xpa);
    if (num_instances > 0) num_instances--;
}



/* Bring up help documentation */
static void
help (BonoboUIComponent *uic, XPenguinsApplet *xpa, const gchar *verbname)
{
    GError *error = NULL;

    gnome_help_display_desktop (NULL, PACKAGE, PACKAGE, NULL, &error);
    if (error)
    {
        g_warning(_("help error: %s"), error->message);
        g_error_free(error);
    }
}



/* Bring up the "About" window */
static void
about (BonoboUIComponent *uic, XPenguinsApplet *xpa, const gchar *verbname)
{
    GtkWidget *about_box;
    static GdkPixbuf *pixbuf;
    static const gchar *authors[] = {
        "Robin Hogan <R.J.Hogan@reading.ac.uk>",
        "Sebastian Henschel <shensche@kodeaffe.de>",
        NULL
    };
    gchar *translators =
        g_ascii_strcasecmp ("translator_credits",
                            _("translator_credits")) != 0 ?
                            _("translator_credits") : NULL;

    /* pixbuf is allowed to be NULL after loading */
    if (!pixbuf)
    {
        gchar *fnam =
            gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_PIXMAP,
                                       PIXMAPDIR "/gnome-xpenguins.png",
                                       TRUE, NULL);
        if (fnam)
        {
            pixbuf = gdk_pixbuf_new_from_file (fnam, NULL);
            g_free (fnam);
        }
    }

    about_box = gnome_about_new (
        _("XPenguins Applet"), VERSION, _("Copyright (C) 2000-2002 "),
        _("Animates a friendly family of penguins in your root window."),
        authors, NULL, translators, pixbuf);
    gtk_widget_show (about_box);
}



/* Display one of the three applet buttons */
static void
make_pixmap_button (GtkWidget **button, GtkWidget **button_pixmap,
                    gchar **xpm, gint width, gint height, guint border)
{
    GdkPixbuf *pixbuf1, *pixbuf2;

    if (*button_pixmap) gtk_widget_destroy (*button_pixmap);

    pixbuf1 = gdk_pixbuf_new_from_xpm_data ((const gchar**) xpm);
    pixbuf2 = gdk_pixbuf_scale_simple (pixbuf1, width - border * 2,
                                       height - border * 2,
                                       GDK_INTERP_NEAREST);
    g_free (pixbuf1);
    *button_pixmap = gtk_image_new_from_pixbuf (pixbuf2);

    gtk_container_add (GTK_CONTAINER (*button), *button_pixmap);
    GTK_WIDGET_UNSET_FLAGS (*button, GTK_CAN_DEFAULT);
    GTK_WIDGET_UNSET_FLAGS (*button, GTK_CAN_FOCUS);
    gtk_widget_set_size_request (GTK_WIDGET (*button), width, height);
    gtk_widget_show (*button_pixmap);
}



/* Create the three buttons in the applet */
static void
change_size_cb (GtkWidget *widget, gint size, gpointer data)
{
    XPenguinsApplet *xpa = data;
    char **my_stop_xpm = stop_xpm;
    char **my_play_pause_xpm = play_pause_xpm;
    guint border = size / 12 - 1;

    if (border < 2) /* A tiny panel */
    {
        my_stop_xpm = stop_tiny_xpm;
        my_play_pause_xpm = play_pause_tiny_xpm;
    }
    else if (border == 2) /* A small panel */
    {
        my_stop_xpm = stop_small_xpm;
        my_play_pause_xpm = play_pause_small_xpm;
    }
    else /* A huge panel */
        border = 4;

    make_pixmap_button (&(xpa->stop), &(xpa->stop_pixmap),
                        my_stop_xpm, (16 * size) / 48,
                        (16 * size) / 48, border);
    make_pixmap_button (&(xpa->playpause), &(xpa->playpause_pixmap),
                        my_play_pause_xpm, (32 * size) / 48,
                        (16 * size) / 48, border);
    make_pixmap_button (&(xpa->button), &(xpa->button_pixmap),
                        sitter_button_xpm, (32 * size) / 48,
                        (32 * size) / 48, 1);
}



static XPenguinsApplet *
create_xpa (PanelApplet *applet)
{
    XPenguinsApplet *xpa;
    GtkTooltips *tooltip = NULL;
    GtkWidget *frame = NULL;

    /* Allocate memory to hold all the applet data */
    xpa = (XPenguinsApplet *) g_new (XPenguinsApplet, 1);

    /* Initialise the applet data */
    xpa->applet = applet;
    xpa->prop_window = NULL;

    xpenguins_verbose = 0;

    xpa->active = FALSE;
    xpa->loaded = FALSE;
    xpa->paused = FALSE;
    xpa->theme_list = NULL;
    xpa->theme_info = NULL;
    xpa->theme_name = NULL;

    xpenguins_ignorepopups ((char) xpa->ignorepopups);

    xpa->vbox = gtk_vbox_new (FALSE,0);
    xpa->hbox = gtk_hbox_new (FALSE,0);

    /* Make the sitting penguin "button", play-pause and stop buttons */
    xpa->button = gtk_event_box_new ();
    tooltip = gtk_tooltips_new ();
    gtk_tooltips_set_tip (tooltip, GTK_WIDGET (xpa->button),
                          _("XPenguins Applet"), NULL);
    /* the frame gives the penguin the button-look */
    frame = gtk_frame_new (NULL);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
    gtk_container_add (GTK_CONTAINER (frame), xpa->button);
    gtk_widget_show (xpa->button);

    xpa->stop = gtk_button_new ();
    tooltip = gtk_tooltips_new ();
    gtk_tooltips_set_tip (tooltip, GTK_WIDGET (xpa->stop),
                          _("Stop Penguins"), NULL);

    xpa->playpause = gtk_button_new ();
    tooltip = gtk_tooltips_new ();
    gtk_tooltips_set_tip (tooltip, GTK_WIDGET (xpa->playpause),
                          _("Start/Pause Penguins"), NULL);

    xpa->button_pixmap = NULL;
    xpa->stop_pixmap = NULL;
    xpa->playpause_pixmap = NULL;

    change_size_cb (NULL, panel_applet_get_size (applet), xpa);

    /* Set the callbacks for actions on the various buttons */
    g_signal_connect (G_OBJECT (xpa->playpause),"clicked",
                      G_CALLBACK (playpause), xpa);
    g_signal_connect (G_OBJECT (xpa->stop),"clicked",
                      G_CALLBACK(stop), xpa);

    /* Pack it all in and show */
    gtk_box_pack_start (GTK_BOX (xpa->hbox),xpa->stop, FALSE, FALSE, 0);
    gtk_widget_show (xpa->stop);
    gtk_box_pack_start (GTK_BOX (xpa->hbox),xpa->playpause, FALSE, FALSE, 0);
    gtk_widget_show (xpa->playpause);
    gtk_box_pack_start (GTK_BOX (xpa->vbox),frame, FALSE, FALSE, 0);
    gtk_widget_show (frame);
    gtk_box_pack_start (GTK_BOX (xpa->vbox),xpa->hbox, FALSE, FALSE, 0);
    gtk_widget_show (xpa->hbox);
    gtk_widget_show (xpa->vbox);

    /* build surrounding frame and pack it into the applet */
    gtk_container_add (GTK_CONTAINER (applet), xpa->vbox);

    return xpa;
}




/* Fills the applet with widgetry and sets up gconf, menu, etc. */
static gboolean
applet_fill (PanelApplet *applet)
{
    XPenguinsApplet *xpa;

    gtk_widget_realize (GTK_WIDGET (applet));
    xpa = create_xpa (applet);

    panel_applet_setup_menu_from_file (applet, NULL, UIDIR "/" PACKAGE ".xml",
                                       NULL, menu_verbs, xpa);
    panel_applet_add_preferences (
        applet, "/schemas/apps/" PACKAGE "/prefs", NULL);
    xpa_property_load (xpa);

    /* Register general applet callbacks */
    g_signal_connect (G_OBJECT (applet), "destroy",
                      G_CALLBACK (destroy), xpa);
    g_signal_connect (G_OBJECT (applet),"change_size",
                      G_CALLBACK (change_size_cb), xpa);
    gtk_widget_show (GTK_WIDGET (applet));

    return TRUE;
}


/* The factory produces another instance of the applet */
static gboolean
applet_factory (PanelApplet *applet, const gchar *iid, gpointer data)
{
    gboolean retval = FALSE;

    if (num_instances > 0)
    {
        GtkWidget *dialog;
        gint response;
        gchar *msg = g_strdup_printf (_("There is already an instance of %s "
                                        "running. XPenguins is not designed "
                                        "to support multiple instances. This "
                                        "will lead to random crashes on some "
                                        "occasions. Be warned, if you still "
                                        "want to continue!"), PACKAGE);

        dialog = gtk_message_dialog_new (NULL,
                                         GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_MESSAGE_WARNING,
                                         GTK_BUTTONS_OK_CANCEL,
                                         msg);
        response = gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        g_free (msg);
        if (response == GTK_RESPONSE_CANCEL) return FALSE;
    }
    num_instances++;

    if (!g_ascii_strcasecmp (iid, "OAFIID:XPenguinsApplet"))
        retval = applet_fill (applet);

    if (!retval) num_instances--;
    return retval;
}


PANEL_APPLET_BONOBO_FACTORY ("OAFIID:XPenguinsApplet_Factory",
                             PANEL_TYPE_APPLET,
                             PACKAGE,
                             VERSION,
                             applet_factory,
                             NULL);

